/*
 * Decompiled with CFR 0.152.
 */
package net.skinsrestorer.shared.storage;

import java.io.File;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.MalformedInputException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Random;
import java.util.TreeMap;
import java.util.concurrent.TimeUnit;
import javax.sql.rowset.CachedRowSet;
import net.skinsrestorer.api.exception.SkinRequestException;
import net.skinsrestorer.api.interfaces.ISkinStorage;
import net.skinsrestorer.api.property.GenericProperty;
import net.skinsrestorer.api.property.IProperty;
import net.skinsrestorer.shared.exception.NotPremiumException;
import net.skinsrestorer.shared.storage.Config;
import net.skinsrestorer.shared.storage.Locale;
import net.skinsrestorer.shared.storage.MySQL;
import net.skinsrestorer.shared.utils.C;
import net.skinsrestorer.shared.utils.connections.MineSkinAPI;
import net.skinsrestorer.shared.utils.connections.MojangAPI;
import net.skinsrestorer.shared.utils.log.SRLogger;

public class SkinStorage
implements ISkinStorage {
    private final SRLogger logger;
    private final MojangAPI mojangAPI;
    private final MineSkinAPI mineSkinAPI;
    private MySQL mysql;
    private File skinsFolder;
    private File playersFolder;

    public void loadFolders(File pluginFolder) {
        this.skinsFolder = new File(pluginFolder, "Skins");
        this.skinsFolder.mkdirs();
        this.playersFolder = new File(pluginFolder, "Players");
        this.playersFolder.mkdirs();
    }

    public void preloadDefaultSkins() {
        if (!Config.DEFAULT_SKINS_ENABLED) {
            return;
        }
        ArrayList toRemove = new ArrayList();
        Config.DEFAULT_SKINS.forEach(skin -> {
            block3: {
                try {
                    if (!C.validUrl(skin)) {
                        this.getSkinForPlayer((String)skin);
                    }
                }
                catch (SkinRequestException e) {
                    toRemove.add(skin);
                    this.logger.warning("[WARNING] DefaultSkin '" + skin + "'(.skin) could not be found or requested! Removing from list..");
                    this.logger.debug("[DEBUG] DefaultSkin '" + skin + "' error: ");
                    if (!Config.DEBUG) break block3;
                    e.printStackTrace();
                }
            }
        });
        Config.DEFAULT_SKINS.removeAll(toRemove);
        if (Config.DEFAULT_SKINS.isEmpty()) {
            this.logger.warning("[WARNING] No more working DefaultSkin left... disabling feature");
            Config.DEFAULT_SKINS_ENABLED = false;
        }
    }

    public IProperty getDefaultSkinForPlayer(String playerName) throws SkinRequestException {
        String skin = this.getDefaultSkinName(playerName);
        if (C.validUrl(skin)) {
            return this.mineSkinAPI.genSkin(skin, null, null);
        }
        return this.getSkinForPlayer(skin);
    }

    @Override
    public IProperty getSkinForPlayer(String playerName) throws SkinRequestException {
        Optional<IProperty> textures;
        Optional<String> skin = this.getSkinName(playerName);
        if (!skin.isPresent()) {
            skin = Optional.of(playerName.toLowerCase());
        }
        if (!(textures = this.getSkinData(skin.get())).isPresent()) {
            try {
                if (!C.validMojangUsername(skin.get())) {
                    throw new SkinRequestException(Locale.INVALID_PLAYER.replace("%player", skin.get()));
                }
                textures = this.mojangAPI.getSkin(skin.get());
                if (!textures.isPresent()) {
                    throw new SkinRequestException(Locale.ERROR_NO_SKIN);
                }
                this.setSkinData(skin.get(), textures.get());
                return textures.get();
            }
            catch (SkinRequestException e) {
                throw e;
            }
            catch (Exception e) {
                e.printStackTrace();
                throw new SkinRequestException(Locale.WAIT_A_MINUTE);
            }
        }
        return textures.get();
    }

    @Override
    public Optional<String> getSkinName(String playerName) {
        playerName = playerName.toLowerCase();
        if (Config.MYSQL_ENABLED) {
            CachedRowSet crs = this.mysql.query("SELECT * FROM " + Config.MYSQL_PLAYER_TABLE + " WHERE Nick=?", playerName);
            if (crs != null) {
                try {
                    String skin = crs.getString("Skin");
                    if (skin.isEmpty()) {
                        this.removeSkin(playerName);
                        return Optional.empty();
                    }
                    return Optional.of(skin);
                }
                catch (Exception e) {
                    e.printStackTrace();
                }
            }
        } else {
            playerName = this.removeForbiddenChars(playerName);
            File playerFile = new File(this.playersFolder, playerName + ".player");
            try {
                if (!playerFile.exists()) {
                    return Optional.empty();
                }
                List<String> lines = Files.readAllLines(playerFile.toPath());
                if (lines.size() < 1) {
                    this.removeSkin(playerName);
                    return Optional.empty();
                }
                return Optional.of(lines.get(0));
            }
            catch (MalformedInputException e) {
                this.removeSkin(playerName);
            }
            catch (IOException e) {
                e.printStackTrace();
            }
        }
        return Optional.empty();
    }

    @Override
    public Optional<IProperty> getSkinData(String skinName) {
        return this.getSkinData(skinName, true);
    }

    public Optional<IProperty> getSkinData(String skinName, boolean updateOutdated) {
        skinName = skinName.toLowerCase();
        if (Config.MYSQL_ENABLED) {
            CachedRowSet crs = this.mysql.query("SELECT * FROM " + Config.MYSQL_SKIN_TABLE + " WHERE Nick=?", skinName);
            if (crs != null) {
                try {
                    String value = crs.getString("Value");
                    String signature = crs.getString("Signature");
                    String timestamp = crs.getString("timestamp");
                    return Optional.of(this.createProperty(skinName, updateOutdated, value, signature, timestamp));
                }
                catch (Exception e) {
                    this.removeSkinData(skinName);
                    this.logger.info("Unsupported skin format.. removing (" + skinName + ").");
                }
            }
        } else {
            skinName = this.removeWhitespaces(skinName);
            skinName = this.removeForbiddenChars(skinName);
            File skinFile = new File(this.skinsFolder, skinName + ".skin");
            try {
                if (!skinFile.exists()) {
                    return Optional.empty();
                }
                List<String> lines = Files.readAllLines(skinFile.toPath());
                String value = lines.get(0);
                String signature = lines.get(1);
                String timestamp = lines.get(2);
                return Optional.of(this.createProperty(skinName, updateOutdated, value, signature, timestamp));
            }
            catch (Exception e) {
                this.removeSkinData(skinName);
                this.logger.info("Unsupported skin format.. removing (" + skinName + ").");
            }
        }
        return Optional.empty();
    }

    private IProperty createProperty(String playerName, boolean updateOutdated, String value, String signature, String timestamp) throws SkinRequestException {
        Optional<IProperty> skin;
        if (updateOutdated && C.validMojangUsername(playerName) && this.isOld(Long.parseLong(timestamp)) && (skin = this.mojangAPI.getSkin(playerName)).isPresent()) {
            this.setSkinData(playerName, skin.get());
            return skin.get();
        }
        return this.mojangAPI.createProperty("textures", value, signature);
    }

    private boolean isOld(long timestamp) {
        if (timestamp == 0L || Config.DISALLOW_AUTO_UPDATE_SKIN) {
            return false;
        }
        return timestamp + TimeUnit.MINUTES.toMillis(Config.SKIN_EXPIRES_AFTER) <= System.currentTimeMillis();
    }

    @Override
    public void removeSkin(String playerName) {
        playerName = playerName.toLowerCase();
        if (Config.MYSQL_ENABLED) {
            this.mysql.execute("DELETE FROM " + Config.MYSQL_PLAYER_TABLE + " WHERE Nick=?", playerName);
        } else {
            playerName = this.removeForbiddenChars(playerName);
            File playerFile = new File(this.playersFolder, playerName + ".player");
            try {
                Files.deleteIfExists(playerFile.toPath());
            }
            catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    public void removeSkinData(String skinName) {
        skinName = skinName.toLowerCase();
        if (Config.MYSQL_ENABLED) {
            this.mysql.execute("DELETE FROM " + Config.MYSQL_SKIN_TABLE + " WHERE Nick=?", skinName);
        } else {
            skinName = this.removeWhitespaces(skinName);
            skinName = this.removeForbiddenChars(skinName);
            File skinFile = new File(this.skinsFolder, skinName + ".skin");
            try {
                Files.deleteIfExists(skinFile.toPath());
            }
            catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    @Override
    public void setSkinName(String playerName, String skinName) {
        playerName = playerName.toLowerCase();
        if (Config.MYSQL_ENABLED) {
            this.mysql.execute("INSERT INTO " + Config.MYSQL_PLAYER_TABLE + " (Nick, Skin) VALUES (?,?) ON DUPLICATE KEY UPDATE Skin=?", playerName, skinName, skinName);
        } else {
            playerName = this.removeForbiddenChars(playerName);
            File playerFile = new File(this.playersFolder, playerName + ".player");
            try {
                if (!playerFile.exists() && !playerFile.createNewFile()) {
                    throw new IOException("Could not create player file!");
                }
                try (OutputStreamWriter writer = new OutputStreamWriter((OutputStream)new FileOutputStream(playerFile), StandardCharsets.UTF_8);){
                    skinName = this.removeWhitespaces(skinName);
                    skinName = this.removeForbiddenChars(skinName);
                    writer.write(skinName);
                }
            }
            catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    public void setSkinData(String skinName, IProperty textures) {
        this.setSkinData(skinName, textures, Long.toString(System.currentTimeMillis()));
    }

    public void setSkinData(String skinName, IProperty textures, String timestamp) {
        skinName = skinName.toLowerCase();
        String value = textures.getValue();
        String signature = textures.getSignature();
        if (Config.MYSQL_ENABLED) {
            this.mysql.execute("INSERT INTO " + Config.MYSQL_SKIN_TABLE + " (Nick, Value, Signature, timestamp) VALUES (?,?,?,?) ON DUPLICATE KEY UPDATE Value=?, Signature=?, timestamp=?", skinName, value, signature, timestamp, value, signature, timestamp);
        } else {
            skinName = this.removeWhitespaces(skinName);
            skinName = this.removeForbiddenChars(skinName);
            File skinFile = new File(this.skinsFolder, skinName + ".skin");
            try {
                if (value.isEmpty() || signature.isEmpty() || timestamp.isEmpty()) {
                    return;
                }
                if (!skinFile.exists() && !skinFile.createNewFile()) {
                    throw new IOException("Could not create skin file!");
                }
                try (OutputStreamWriter writer = new OutputStreamWriter((OutputStream)new FileOutputStream(skinFile), StandardCharsets.UTF_8);){
                    writer.write(value + "\n" + signature + "\n" + timestamp);
                }
            }
            catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    public Map<String, IProperty> getSkins(int number) {
        TreeMap<String, IProperty> list = new TreeMap<String, IProperty>();
        if (Config.MYSQL_ENABLED) {
            String filterBy = "";
            String orderBy = "Nick";
            if (Config.CUSTOM_GUI_ENABLED) {
                StringBuilder sb = new StringBuilder();
                if (Config.CUSTOM_GUI_ONLY) {
                    Config.CUSTOM_GUI_SKINS.forEach(sb.append("|")::append);
                    filterBy = "WHERE Nick RLIKE '" + sb.substring(1) + "'";
                } else {
                    Config.CUSTOM_GUI_SKINS.forEach(skin -> sb.append(", '").append((String)skin).append("'"));
                    orderBy = "FIELD(Nick" + sb + ") DESC, Nick";
                }
            }
            CachedRowSet crs = this.mysql.query("SELECT Nick, Value, Signature FROM " + Config.MYSQL_SKIN_TABLE + " " + filterBy + " ORDER BY " + orderBy, new Object[0]);
            int i = 0;
            try {
                do {
                    if (i >= number) {
                        list.put(crs.getString("Nick").toLowerCase(), this.mojangAPI.createProperty("textures", crs.getString("Value"), crs.getString("Signature")));
                    }
                    ++i;
                } while (crs.next());
            }
            catch (SQLException sQLException) {}
        } else {
            FilenameFilter skinFileFilter = (dir, name) -> name.endsWith(".skin");
            Object[] fileNames = this.skinsFolder.list(skinFileFilter);
            if (fileNames == null) {
                return list;
            }
            Arrays.sort(fileNames);
            int i = 0;
            for (Object file : fileNames) {
                String skinName = ((String)file).replace(".skin", "");
                if (i >= number) {
                    if (Config.CUSTOM_GUI_ONLY) {
                        for (String GuiSkins : Config.CUSTOM_GUI_SKINS) {
                            if (!skinName.toLowerCase().contains(GuiSkins.toLowerCase())) continue;
                            this.getSkinData(skinName, false).ifPresent(property -> list.put(skinName.toLowerCase(), (IProperty)property));
                        }
                    } else {
                        this.getSkinData(skinName, false).ifPresent(property -> list.put(skinName.toLowerCase(), (IProperty)property));
                    }
                }
                ++i;
            }
        }
        return list;
    }

    public Map<String, GenericProperty> getSkinsRaw(int number) {
        TreeMap<String, GenericProperty> list = new TreeMap<String, GenericProperty>();
        if (Config.MYSQL_ENABLED) {
            CachedRowSet crs = this.mysql.query("SELECT Nick, Value, Signature FROM " + Config.MYSQL_SKIN_TABLE + " ORDER BY `Nick`", new Object[0]);
            int i = 0;
            int foundSkins = 0;
            try {
                do {
                    if (i >= number && foundSkins <= 26) {
                        GenericProperty prop = new GenericProperty();
                        prop.setName("textures");
                        prop.setValue(crs.getString("Value"));
                        prop.setSignature(crs.getString("Signature"));
                        list.put(crs.getString("Nick"), prop);
                        ++foundSkins;
                    }
                    ++i;
                } while (crs.next());
            }
            catch (SQLException prop) {}
        } else {
            FilenameFilter skinFileFilter = (dir, name) -> name.endsWith(".skin");
            Object[] fileNames = this.skinsFolder.list(skinFileFilter);
            if (fileNames == null) {
                return list;
            }
            Arrays.sort(fileNames);
            int i = 0;
            int foundSkins = 0;
            for (Object file : fileNames) {
                String skinName = ((String)file).replace(".skin", "");
                File skinFile = new File(this.skinsFolder, (String)file);
                if (i >= number && foundSkins <= 26) {
                    try {
                        if (!skinFile.exists()) {
                            return null;
                        }
                        List<String> lines = Files.readAllLines(skinFile.toPath());
                        GenericProperty prop = new GenericProperty();
                        prop.setName("textures");
                        prop.setValue(lines.get(0));
                        prop.setSignature(lines.get(1));
                        list.put(skinName, prop);
                        ++foundSkins;
                    }
                    catch (Exception exception) {
                        // empty catch block
                    }
                }
                ++i;
            }
        }
        return list;
    }

    public boolean updateSkinData(String skinName) throws SkinRequestException {
        if (!C.validMojangUsername(skinName)) {
            throw new SkinRequestException(Locale.ERROR_UPDATING_CUSTOMSKIN);
        }
        boolean updateDisabled = false;
        if (Config.MYSQL_ENABLED) {
            CachedRowSet crs = this.mysql.query("SELECT timestamp FROM " + Config.MYSQL_SKIN_TABLE + " WHERE Nick=?", skinName);
            if (crs != null) {
                try {
                    updateDisabled = crs.getString("timestamp").equals("0");
                }
                catch (Exception exception) {}
            }
        } else {
            skinName = this.removeWhitespaces(skinName);
            skinName = this.removeForbiddenChars(skinName);
            File skinFile = new File(this.skinsFolder, skinName + ".skin");
            try {
                if (skinFile.exists()) {
                    updateDisabled = Files.readAllLines(skinFile.toPath()).get(2).equals("0");
                }
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
        if (updateDisabled) {
            throw new SkinRequestException(Locale.ERROR_UPDATING_CUSTOMSKIN);
        }
        try {
            Optional<IProperty> textures;
            Optional<String> mojangUUID = this.mojangAPI.getUUIDMojang(skinName);
            if (mojangUUID.isPresent() && (textures = this.mojangAPI.getProfileMojang(mojangUUID.get())).isPresent()) {
                this.setSkinData(skinName, textures.get());
                return true;
            }
        }
        catch (NotPremiumException e) {
            throw new SkinRequestException(Locale.ERROR_UPDATING_CUSTOMSKIN);
        }
        return false;
    }

    public String getDefaultSkinName(String playerName) {
        return this.getDefaultSkinName(playerName, false);
    }

    public String getDefaultSkinName(String playerName, boolean clear) {
        playerName = playerName.replaceAll("^\\\\s+", "");
        playerName = playerName.replaceAll("\\\\s+$", "");
        if (Config.DEFAULT_SKINS_ENABLED) {
            if (!Config.DEFAULT_SKINS_PREMIUM) {
                try {
                    if (C.validMojangUsername(playerName) && this.mojangAPI.getUUID(playerName) != null) {
                        return playerName;
                    }
                }
                catch (SkinRequestException skinRequestException) {
                    // empty catch block
                }
            }
            if (!this.getSkinName(playerName).isPresent() || clear) {
                List<String> skins = Config.DEFAULT_SKINS;
                String randomSkin = skins.size() > 1 ? skins.get(new Random().nextInt(skins.size())) : skins.get(0);
                return randomSkin != null ? randomSkin : playerName;
            }
        }
        if (clear) {
            return playerName;
        }
        return this.getSkinName(playerName).orElse(playerName);
    }

    private String removeForbiddenChars(String str) {
        return str.replaceAll("[\\\\/:*?\"<>|]", "\u00b7");
    }

    private String removeWhitespaces(String str) {
        if (str.startsWith(" ")) {
            return str;
        }
        return str.replaceAll("\\s", "");
    }

    public SkinStorage(SRLogger logger, MojangAPI mojangAPI, MineSkinAPI mineSkinAPI) {
        this.logger = logger;
        this.mojangAPI = mojangAPI;
        this.mineSkinAPI = mineSkinAPI;
    }

    public void setMysql(MySQL mysql) {
        this.mysql = mysql;
    }
}

